﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Threading;
using System.Runtime.InteropServices;
using System.IO;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.ComponentModel;

namespace MutexFun
{
    /// <summary>
    /// Klasa, która przesyła obiekty za pośrednictwem wspólnej pamięci używając muteks,
    /// aby zapewnić synchronizację dostępu do wspólnego pamięci.
    /// </summary>
    public class SharedMemoryManager<TransferItemType> : IDisposable 
    {
	    #region Stałe
	    const int INVALID_HANDLE_VALUE = -1;
	    const int FILE_MAP_WRITE = 0x0002;
	    /// <summary>
	    /// Definicja z Win32 API
	    /// </summary>
	    const int ERROR_ALREADY_EXISTS = 183;
	    #endregion

	    #region Składowe prywatne
	    IntPtr _handleFileMapping = IntPtr.Zero;
	    IntPtr _ptrToMemory = IntPtr.Zero;
	    uint _memRegionSize = 0;
	    string _memoryRegionName;
        bool disposed = false;
        int _sharedMemoryBaseSize = 0;
	    Mutex _mtxSharedMem = null;

	    #endregion

        #region Tworzenie / Czyszczenie
        public SharedMemoryManager(string name,int sharedMemoryBaseSize)
        {
            // można tworzyć tylko dla obiektów serializowanych
            if (!typeof(TransferItemType).IsSerializable)
                throw new ArgumentException(
                    string.Format("Obiektu {0} nie można serializować.",
                        typeof(TransferItemType)));

            if (string.IsNullOrEmpty(name))
                throw new ArgumentNullException("name");

            if (sharedMemoryBaseSize <= 0)
                throw new ArgumentOutOfRangeException("sharedMemoryBaseSize",
                    "Bazowy rozmiar wspólnej pamięci musi być wartością większą od zera");

            // ustawienie nazwy regionu
            _memoryRegionName = name;
            // zapisanie rozmiaru bazowego
            _sharedMemoryBaseSize = sharedMemoryBaseSize;
            // ustawienie rozmiaru regionu pamięci
            _memRegionSize = (uint)(_sharedMemoryBaseSize + sizeof(int));
            // ustawienie sekcji pamięci wspólnej
            SetupSharedMemory();
        }

        private void SetupSharedMemory()
        {
            // zajęcie przestrzeni w pliku stronicującym
            _handleFileMapping =
                PInvoke.CreateFileMapping((IntPtr)INVALID_HANDLE_VALUE,
                                    IntPtr.Zero,
                                    PInvoke.PageProtection.ReadWrite,
                                    0,
                                    _memRegionSize,
                                    _memoryRegionName);

            if (_handleFileMapping == IntPtr.Zero)
            {
                throw new Win32Exception(
                    "Nie udało się utworzyć odwzorowania pliku");
            }

            // sprawdzenie statusu błędu
            int retVal = Marshal.GetLastWin32Error();
            if (retVal == ERROR_ALREADY_EXISTS)
            {
                // otwarto region, który już istnieje
                // muteks nie będzie początkowym właścicielem,
                // ponieważ podłączamy się do już istniejącego regionu
                _mtxSharedMem = new Mutex(false,
                    string.Format("{0}mtx{1}", 
                        typeof(TransferItemType), _memoryRegionName));
            }
            else if (retVal == 0)
            {
                // otwarto nowy region
                // muteks będzie jego początkowym właścicielem
                _mtxSharedMem = new Mutex(true,
                    string.Format("{0}mtx{1}", 
                        typeof(TransferItemType), _memoryRegionName));
            }
            else
            {
                // coś innego poszło nie tak..
                throw new Win32Exception(retVal, "Błąd w trakcie tworzenia odwzorowania pliku");
            }

            // odwzorowanie wspólnej pamięci
            _ptrToMemory = PInvoke.MapViewOfFile(_handleFileMapping,
                                            FILE_MAP_WRITE,
                                            0, 0, IntPtr.Zero);

            if (_ptrToMemory == IntPtr.Zero)
            {
                retVal = Marshal.GetLastWin32Error();
                throw new Win32Exception(retVal, "Nie udało się odwzorować widoku pliku");
            }

            retVal = Marshal.GetLastWin32Error();
            if (retVal != 0 && retVal != ERROR_ALREADY_EXISTS)
            {
                // coś innego poszło nie tak..
                throw new Win32Exception(retVal, "Nie udało się odwzorować widoku pliku");
            }
        }

	    ~SharedMemoryManager()
	    {
		    // zamknięcie
		    Dispose(false);
	    }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        private void Dispose(bool disposing)
        {
            // sprawdzenie, czy metoda Dispose została już wywołana
            if (!this.disposed)
            {
                CloseSharedMemory();
            }
            disposed = true;
        }

        private void CloseSharedMemory()
        {
            if (_ptrToMemory != IntPtr.Zero)
            {
                // zamknięcie odwzorowania wspólnej pamięci
                PInvoke.UnmapViewOfFile(_ptrToMemory);
                _ptrToMemory = IntPtr.Zero;
            }
            if (_handleFileMapping != IntPtr.Zero)
            {
                // zamknięcie uchwytu 
                PInvoke.CloseHandle(_handleFileMapping);
                _handleFileMapping = IntPtr.Zero;
            }
        }

        public void Close()
        {
            CloseSharedMemory();
        }
        #endregion

        #region Właściwości
        public int SharedMemoryBaseSize
        {
            get { return _sharedMemoryBaseSize; } 
        }
        #endregion

        #region Metody publiczne
        /// <summary>
	    /// Wysłanie serializowanego obiektu za pośrednictwem wspólnej pamięci
	    /// i oczekiwanie na jego odebranie
	    /// </summary>
	    /// <param name="transferObject"></param>
        public void SendObject(TransferItemType transferObject)
        {
	        // utworzenie strumienia pamięci, zainicjowanie rozmiaru
            using (MemoryStream ms = new MemoryStream())
            {
                // pobranie obiektu formatującego do użycia w trakcie serializacji
                BinaryFormatter formatter = new BinaryFormatter();
                try
                {
                    // zserializowanie obiektu do strumienia
                    formatter.Serialize(ms, transferObject);

                    // odczytanie bajtów dla zserializowanego obiektu
                    byte[] bytes = ms.ToArray();

                    // sprawdzenie, czy obiekt się zmieści
                    if(bytes.Length + sizeof(int) > _memRegionSize)
                    {
                        string fmt = 
                            "Egzemplarz obiektu {0} zserializowany w {1} bajtach " +
                            "jest zbyt duży dla regionu wspólnej pamięci";
                        
                        string msg = 
                            string.Format(fmt,
                                typeof(TransferItemType),bytes.Length);

                        throw new ArgumentException(msg, "transferObject");
                    }

                    // wypisanie długości obiektu
                    Marshal.WriteInt32(this._ptrToMemory, bytes.Length);

                    // wypisanie bajtów
                    Marshal.Copy(bytes, 0, this._ptrToMemory, bytes.Length);
                }
                finally
                {
                    // zasygnalizowanie innemu procesowi używającemu muteksu,
                    // że może rozpocząć odbieranie
                    _mtxSharedMem.ReleaseMutex();

                    // oczekiwanie na sygnał od innego procesu,
                    // że obiekt został odebrany i można kontynuować przetwarzanie
                    _mtxSharedMem.WaitOne();
                }
            }
        }

	    /// <summary>
	    /// Oczekiwanie na dotarcie obiektu do wspólnej pamięci i jego deserializacja
	    /// </summary>
	    /// <returns>przekazany obiekt</returns>
        public TransferItemType ReceiveObject()
        {
	        // oczekiwanie na zakolejkowanie przez nadawcę muteksu dla obiektu
	        _mtxSharedMem.WaitOne();

	        // sprawdzenie ilości danych znajdujących się we wspólnej pamięci
	        int count = Marshal.ReadInt32(_ptrToMemory);
	        if (count <= 0)
	        {
		        throw new InvalidDataException("Brak obiektu do odczytania");
	        }

	        // utworzenie tablicy przechowującej bajty
	        byte[] bytes = new byte[count];

	        // odczytanie bajtów obiektu
            Marshal.Copy(_ptrToMemory, bytes, 0, count);

	        // ustawienie strumienia pamięci z bajtami obiektu
            using (MemoryStream ms = new MemoryStream(bytes))
            {
                // ustawienie binarnego obiektu formatującego
                BinaryFormatter formatter = new BinaryFormatter();

                // pobranie obiektu, który zostanie zwrócony
                TransferItemType item;
                try
                {
                    item = (TransferItemType)formatter.Deserialize(ms);
                }
                finally
                {
                    // sygnał o odebraniu obiektu przez zwolnienie muteksu
                    _mtxSharedMem.ReleaseMutex();
                }
                // zwrócenie obiektu
                return item;
            }
        }
	    #endregion
    }

    public class PInvoke
    {
        #region PInvoke defines
        [Flags]
        public enum PageProtection : uint
        {
            NoAccess = 0x01,
            Readonly = 0x02,
            ReadWrite = 0x04,
            WriteCopy = 0x08,
            Execute = 0x10,
            ExecuteRead = 0x20,
            ExecuteReadWrite = 0x40,
            ExecuteWriteCopy = 0x80,
            Guard = 0x100,
            NoCache = 0x200,
            WriteCombine = 0x400,
        }

        [DllImport("kernel32.dll", SetLastError = true)]
        public static extern IntPtr CreateFileMapping(IntPtr hFile,
            IntPtr lpFileMappingAttributes, PageProtection flProtect,
            uint dwMaximumSizeHigh,
            uint dwMaximumSizeLow, string lpName);

        [DllImport("kernel32.dll", SetLastError = true)]
        public static extern IntPtr MapViewOfFile(IntPtr hFileMappingObject, uint
            dwDesiredAccess, uint dwFileOffsetHigh, uint dwFileOffsetLow,
            IntPtr dwNumberOfBytesToMap);

        [DllImport("kernel32.dll", SetLastError = true)]
        public static extern bool UnmapViewOfFile(IntPtr lpBaseAddress);

        [DllImport("kernel32.dll", SetLastError = true)]
        public static extern bool CloseHandle(IntPtr hObject);
        #endregion
    }
}
